<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Kingsway WooCommerce Support Class
 *
 * 处理 WooCommerce 相关功能，包括产品数据面板和前台渲染
 *
 * @class Kingsway_WooCommerce
 * @version 1.0.6
 * @package Kingsway_Plugin
 */
class Kingsway_WooCommerce {

    /**
     * 构造函数，注册 WooCommerce 相关钩子
     */
    public function __construct() {
        add_filter('woocommerce_product_data_tabs', array($this, 'add_product_data_tab'));
        add_action('woocommerce_product_data_panels', array($this, 'product_data_panel'));
        add_action('woocommerce_process_product_meta', array($this, 'save_product_meta'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        add_action('woocommerce_single_product_summary', array($this, 'render_frontend_video'), 21);
        add_action('wp_enqueue_scripts', array($this, 'frontend_enqueue_scripts'));
    }

    /**
     * 检查 WooCommerce 是否激活
     *
     * @return bool
     */
    private function is_woocommerce_active() {
        return class_exists('WooCommerce');
    }

    /**
     * 注册 WooCommerce 产品数据 Tab
     *
     * @param array $tabs
     * @return array
     */
    public function add_product_data_tab($tabs) {
        if (!$this->is_woocommerce_active()) {
            return $tabs;
        }

        $tabs['kingsway_video'] = array(
            'label'    => __('Kingsway Video', 'kingsway-plugin'),
            'target'   => 'kingsway_video_product_data',
            'class'    => array(),
            'priority' => 80,
        );

        return $tabs;
    }

    /**
     * 输出 WooCommerce 产品数据面板内容
     */
    public function product_data_panel() {
        if (!$this->is_woocommerce_active()) {
            return;
        }

        global $post;

        $embed_type   = get_post_meta($post->ID, '_kingsway_embed_type', true);
        $html_content = get_post_meta($post->ID, '_kingsway_html_content', true);
        $item_title   = get_post_meta($post->ID, '_kingsway_item_title', true);

        ?>
        <div id="kingsway_video_product_data" class="panel woocommerce_options_panel hidden">
            <div class="options_group">
                <p class="form-field kingsway-radio-group">
                    <label><?php esc_html_e('Resource Type', 'kingsway-plugin'); ?></label>
                    <span class="kingsway-radio-options">
                        <label>
                            <input type="radio" name="kingsway_resource_type" value="video" <?php checked($embed_type, 'video'); ?> />
                            <?php esc_html_e('Single Video', 'kingsway-plugin'); ?>
                        </label>
                        <label>
                            <input type="radio" name="kingsway_resource_type" value="widget" <?php checked($embed_type, 'widget'); ?> />
                            <?php esc_html_e('Video Widget', 'kingsway-plugin'); ?>
                        </label>
                    </span>
                </p>

                <p class="form-field">
                    <label><?php esc_html_e('Current Association', 'kingsway-plugin'); ?></label>
                    <span id="kingsway-current-association" class="kingsway-current-association">
                        <?php
                        if ($item_title) {
                            echo esc_html($item_title);
                        } elseif ($html_content) {
                            esc_html_e('Selected', 'kingsway-plugin');
                        } else {
                            esc_html_e('(None)', 'kingsway-plugin');
                        }
                        ?>
                    </span>
                </p>

                <input type="hidden" id="kingsway_embed_type" name="kingsway_embed_type" value="<?php echo esc_attr($embed_type); ?>" />
                <input type="hidden" id="kingsway_item_title" name="kingsway_item_title" value="<?php echo esc_attr($item_title); ?>" />
                <textarea style="display:none" id="kingsway_html_content" name="kingsway_html_content"><?php echo esc_textarea($html_content); ?></textarea>

                <p class="form-field">
                    <a href="#" class="button button-primary" id="kingsway-open-library">
                        <?php esc_html_e('Browse Library', 'kingsway-plugin'); ?>
                    </a>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * 保存 WooCommerce 产品的 Kingsway 元数据
     *
     * @param int $post_id
     */
    public function save_product_meta($post_id) {
        if (isset($_POST['kingsway_embed_type'])) {
            update_post_meta($post_id, '_kingsway_embed_type', sanitize_text_field(wp_unslash($_POST['kingsway_embed_type'])));
        }

        if (isset($_POST['kingsway_item_title'])) {
            update_post_meta($post_id, '_kingsway_item_title', sanitize_text_field(wp_unslash($_POST['kingsway_item_title'])));
        }

        if (isset($_POST['kingsway_html_content'])) {
            // 保存完整的 Kingsway HTML（包含 script），与 Gutenberg / Elementor 行为保持一致
            // 这里是管理员操作，允许嵌入脚本
            update_post_meta($post_id, '_kingsway_html_content', wp_unslash($_POST['kingsway_html_content']));
        }
    }

    /**
     * 在 WooCommerce 产品编辑页加载 Kingsway 脚本和样式
     *
     * @param string $hook_suffix
     */
    public function admin_enqueue_scripts($hook_suffix) {
        if (!$this->is_woocommerce_active()) {
            return;
        }

        $screen = get_current_screen();
        if (!$screen || 'product' !== $screen->post_type) {
            return;
        }

        // 共用库脚本
        wp_enqueue_script('kingsway-library');

        // 共用样式
        wp_enqueue_style('kingsway-shared');

        // WooCommerce 后台交互脚本
        wp_enqueue_script(
            'kingsway-woocommerce-admin',
            KINGSWAY_URL . 'assets/js/kingsway-woocommerce-admin.js',
            array('jquery', 'kingsway-library'),
            KINGSWAY_VERSION,
            true
        );

        // 后台样式
        wp_enqueue_style(
            'kingsway-woocommerce-admin',
            KINGSWAY_URL . 'assets/css/kingsway-woocommerce-admin.css',
            array('kingsway-shared'),
            KINGSWAY_VERSION
        );

        // 传递 REST API 信息
        wp_localize_script('kingsway-woocommerce-admin', 'kingswayPlugin', array(
            'apiRoot' => esc_url_raw(rest_url('kingsway/v1/')),
            'nonce'   => wp_create_nonce('wp_rest'),
            'env'     => KINGSWAY_ENV,
        ));
    }

    /**
     * 在前台商品详情页渲染 Kingsway 视频/小部件
     */
    public function render_frontend_video() {
        if (!$this->is_woocommerce_active() || !is_product()) {
            return;
        }

        global $post;
        if (!$post) {
            return;
        }

        $embed_type    = get_post_meta($post->ID, '_kingsway_embed_type', true);
        $html_content  = get_post_meta($post->ID, '_kingsway_html_content', true);

        if (empty($html_content)) {
            return;
        }

        echo '<div class="kingsway-product-video kingsway-embed-type-' . esc_attr($embed_type) . '">';
        // 存的是完整的 Kingsway HTML（包含脚本），与 Gutenberg 一致，直接输出即可
        echo $html_content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        echo '</div>';
    }

    /**
     * 在单个商品页加载 Kingsway 前台脚本
     */
    public function frontend_enqueue_scripts() {
        if (!$this->is_woocommerce_active() || !is_product()) {
            return;
        }

        global $post;
        if (!$post) {
            return;
        }

        $html_content = get_post_meta($post->ID, '_kingsway_html_content', true);
        if (empty($html_content)) {
            return;
        }

        wp_enqueue_script('kingsway-library');
    }
}
