<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Kingsway REST API Class
 *
 * 处理插件的 REST API 端点注册和请求处理
 *
 * @class Kingsway_Rest_API
 * @version 1.0.6
 * @package Kingsway_Plugin
 */
class Kingsway_Rest_API {

    /**
     * 注册 REST API 端点
     */
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }

    /**
     * 注册 REST API 路由
     */
    public function register_rest_routes() {
        // 获取 API key 的端点
        register_rest_route('kingsway/v1', '/api-key/', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_api_key'),
            'permission_callback' => array($this, 'check_permissions'),
        ));

        // 获取视频小部件列表的端点
        register_rest_route('kingsway/v1', '/video-widgets/', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_video_widgets'),
            'permission_callback' => array($this, 'check_permissions'),
        ));
    }

    /**
     * 检查用户权限
     *
     * @return bool
     */
    public function check_permissions() {
        return current_user_can('edit_posts');
    }

    /**
     * 获取 API key
     *
     * @return WP_REST_Response
     */
    public function get_api_key() {
        $api_key = get_option('kingsway-wordpress-plugin-api-key', '');
        return new WP_REST_Response(array('api_key' => $api_key), 200);
    }

    /**
     * 获取视频小部件列表
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_video_widgets($request) {
        $api_key = get_option('kingsway-wordpress-plugin-api-key', '');
        if (empty($api_key)) {
            return new WP_Error('no_api_key', 'API key is not set', array('status' => 400));
        }

        $page = $request->get_param('page') ?: 1;
        $size = $request->get_param('size') ?: 10;
        
        $plugin = Kingsway_Plugin::get_instance();
        $api_host = $plugin->get_api_host();
        
        $url = $api_host . '/video-list-extended/list?page=' . $page . '&size=' . $size;
        
        $response = wp_remote_get($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
            ),
        ));

        if (is_wp_error($response)) {
            return new WP_Error('api_error', $response->get_error_message(), array('status' => 500));
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return new WP_REST_Response($data, 200);
    }
}
