<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Kingsway WordPress Plugin Main Class
 *
 * 主插件类，负责插件的初始化、资源管理和模块加载
 *
 * @class Kingsway_Plugin
 * @version 1.0.6
 * @package Kingsway_Plugin
 */
class Kingsway_Plugin {

    /**
     * 插件实例
     *
     * @var Kingsway_Plugin
     */
    private static $instance = null;

    /**
     * 插件版本
     *
     * @var string
     */
    public $version = '1.0.6';

    /**
     * 插件目录路径
     *
     * @var string
     */
    public $plugin_dir = '';

    /**
     * 插件 URL
     *
     * @var string
     */
    public $plugin_url = '';

    /**
     * 插件配置
     *
     * @var array
     */
    public $config = array();

    /**
     * 私有构造函数，防止直接实例化
     */
    private function __construct() {
        $this->initialize_constants();
        $this->load_config();
        $this->includes();
        $this->init_hooks();
    }

    /**
     * 获取插件实例
     *
     * 实现单例模式，确保只有一个插件实例
     *
     * @return Kingsway_Plugin
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * 初始化常量
     */
    private function initialize_constants() {
        $this->plugin_dir = plugin_dir_path(KINGSWAY_PLUGIN_FILE);
        $this->plugin_url = plugin_dir_url(KINGSWAY_PLUGIN_FILE);

        // 定义插件常量
        define('KINGSWAY_VERSION', $this->version);
        define('KINGSWAY_DIR', $this->plugin_dir);
        define('KINGSWAY_URL', $this->plugin_url);
    }

    /**
     * 加载配置文件
     */
    private function load_config() {
        // 默认配置
        $this->config = array(
            'env' => 'production',
        );

        // 加载配置文件
        $config_file = $this->plugin_dir . 'kingsway-config.php';
        if (file_exists($config_file)) {
            $file_config = include $config_file;
            if (is_array($file_config)) {
                $this->config = array_merge($this->config, $file_config);
            }
        }

        // 定义环境常量
        if (!defined('KINGSWAY_ENV')) {
            define('KINGSWAY_ENV', isset($this->config['env']) ? $this->config['env'] : 'production');
        }
    }

    /**
     * 包含必要的文件并实例化支持类
     */
    private function includes() {
        // 加载支持类
        require_once $this->plugin_dir . 'includes/class-kingsway-rest-api.php';
        require_once $this->plugin_dir . 'includes/class-kingsway-settings.php';
        require_once $this->plugin_dir . 'includes/class-kingsway-gutenberg.php';
        require_once $this->plugin_dir . 'includes/class-kingsway-elementor.php';
        require_once $this->plugin_dir . 'includes/class-kingsway-woocommerce.php';
        
        // 实例化所有支持类
        new Kingsway_Rest_API();
        new Kingsway_Settings();
        new Kingsway_Gutenberg();
        new Kingsway_Elementor();
        new Kingsway_WooCommerce();
    }

    /**
     * 初始化钩子
     */
    private function init_hooks() {
        // 插件激活和停用钩子
        register_activation_hook(KINGSWAY_PLUGIN_FILE, array($this, 'activate'));
        register_deactivation_hook(KINGSWAY_PLUGIN_FILE, array($this, 'deactivate'));

        // 加载文本域
        add_action('init', array($this, 'load_textdomain'));

        // 注册公共脚本
        add_action('init', array($this, 'register_public_scripts'));
    }

    /**
     * 插件激活处理
     */
    public function activate() {
        // 初始化插件设置
        $this->init_settings();
    }

    /**
     * 插件停用处理
     */
    public function deactivate() {
        // 清理插件数据（如果需要）
    }

    /**
     * 初始化设置
     */
    private function init_settings() {
        // 设置默认值
        if (false === get_option('kingsway-wordpress-plugin-api-key')) {
            update_option('kingsway-wordpress-plugin-api-key', '');
        }
    }

    /**
     * 加载文本域
     */
    public function load_textdomain() {
        load_plugin_textdomain('kingsway-plugin', false, dirname(plugin_basename(KINGSWAY_PLUGIN_FILE)) . '/languages/');
    }

    /**
     * 注册公共脚本和样式
     */
    public function register_public_scripts() {
        // 注册共用库脚本
        wp_register_script(
            'kingsway-library',
            $this->plugin_url . 'assets/js/kingsway-library.js',
            array(),
            $this->version,
            true
        );
        
        // 本地化脚本
        wp_localize_script('kingsway-library', 'kingswayPlugin', array(
            'apiRoot' => esc_url_raw(rest_url('kingsway/v1/')),
            'nonce' => wp_create_nonce('wp_rest'),
            'env' => KINGSWAY_ENV,
        ));
        
        // 注册共用样式
        wp_register_style(
            'kingsway-shared',
            $this->plugin_url . 'assets/css/kingsway-shared.css',
            array(),
            $this->version
        );
    }

    /**
     * 获取 API 主机地址
     *
     * 根据当前环境返回相应的 API 主机地址
     *
     * @return string
     */
    public function get_api_host() {
        $env = KINGSWAY_ENV;
        $api_host = 'https://' . ($env === 'development' ? 'test-' : ($env === 'preRelease' ? 'pre-' : '')) . 'api.kingswayvideo.com';
        return $api_host;
    }
}
