<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Kingsway Elementor Support Class
 *
 * 处理 Elementor 小部件的注册和资源加载
 *
 * @class Kingsway_Elementor
 * @version 1.0.6
 * @package Kingsway_Plugin
 */
class Kingsway_Elementor {

    /**
     * 构造函数，注册 Elementor 相关钩子
     */
    public function __construct() {
        add_action('plugins_loaded', array($this, 'init_elementor'), 20);
    }

    public function force_elementor_frontend_js_early() {
        // 跳过编辑器环境，只在前端预览时执行
        // 检查是否在 Elementor 编辑器页面
        if (isset($_GET['action']) && $_GET['action'] === 'elementor') {
            return;
        }
        
        // 检查是否在 Elementor 编辑器 iframe 中
        if (isset($_GET['elementor-preview']) && isset($_GET['ver'])) {
            return;
        }
        
        // 检查 Elementor 编辑器模式（如果可用）
        if (class_exists('\Elementor\Plugin') && \Elementor\Plugin::$instance->editor->is_edit_mode()) {
            return;
        }
        
        if (defined('ELEMENTOR_VERSION')) {
            wp_deregister_script('elementor-frontend');
            
            wp_register_script
            (
                'elementor-frontend', 
                ELEMENTOR_ASSETS_URL . 'js/frontend.min.js', 
                ['jquery'],
                ELEMENTOR_VERSION,
                true,
            );
            
            wp_enqueue_script('elementor-frontend');
        }
    }


    /**
     * 初始化 Elementor 支持
     */
    public function init_elementor() {
        // Check for required PHP version
        if (version_compare(PHP_VERSION, '7.0', '<')) {
            add_action('admin_notices', array($this, 'admin_notice_minimum_php_version'));
            return;
        }

        // Check for required Elementor version
        if (!did_action('elementor/loaded')) {
            return; // Elementor未安装，不显示错误
        }

        if (defined('ELEMENTOR_VERSION') && version_compare(ELEMENTOR_VERSION, '3.0.0', '<')) {
            add_action('admin_notices', array($this, 'admin_notice_minimum_elementor_version'));
            return;
        }

        

        // Add Plugin actions
        add_action('elementor/widgets/register', array($this, 'register_elementor_widgets'));
        add_action('elementor/frontend/after_enqueue_styles', array($this, 'elementor_styles'));
        // 使用 wp_enqueue_scripts 钩子，在更早的时机加载 elementor-frontend.js
        add_action('wp_enqueue_scripts', array($this, 'force_elementor_frontend_js_early'), 1);
        add_action('wp_enqueue_scripts', array($this, 'elementor_scripts'), 2);
    }

    /**
     * 注册 Elementor Widgets
     *
     * @param ElementorWidgets_Manager $widgets_manager
     */
    public function register_elementor_widgets($widgets_manager) {
        if (file_exists(KINGSWAY_DIR . 'elementor/kingsway-video-widget.php')) {
            require_once(KINGSWAY_DIR . 'elementor/kingsway-video-widget.php');
            $widgets_manager->register(new Kingsway_Video_Widget());
        }
    }

    /**
     * Elementor Widget Styles
     */
    public function elementor_styles() {
        // 加载共用样式
        wp_enqueue_style('kingsway-shared');
        
        // 加载 Elementor 特定样式
        wp_register_style('kingsway-elementor', KINGSWAY_URL . 'elementor/assets/css/kingsway-elementor.css', array('kingsway-shared'), KINGSWAY_VERSION);
        wp_enqueue_style('kingsway-elementor');
    }

    /**
     * Elementor Widget Scripts
     */
    public function elementor_scripts() {
        wp_enqueue_script('kingsway-library');
        
        // 移除对 elementor-frontend 的硬依赖，改为在运行时检查
        // JavaScript 端会检查 Elementor 是否可用，如果不可用会直接返回，不会执行任何代码
        wp_register_script('kingsway-elementor', KINGSWAY_URL . 'elementor/assets/js/kingsway-elementor.js', ['jquery', 'kingsway-library'], KINGSWAY_VERSION, true);
        
        // 传递API设置到前端
        wp_localize_script('kingsway-elementor', 'kingswayPlugin', array(
            'apiRoot' => esc_url_raw(rest_url('kingsway/v1/')),
            'nonce' => wp_create_nonce('wp_rest'),
            'cssUrl' => KINGSWAY_URL . 'elementor/assets/css/kingsway-elementor.css',
            'env'   => KINGSWAY_ENV,
        ));
        
        // 直接加载脚本，让 JavaScript 端处理所有检查
        // JavaScript 已经添加了 isElementorAvailable() 检查，如果 Elementor 不可用会直接返回
        wp_enqueue_script('kingsway-elementor');

        // 在编辑器中也加载脚本
        add_action('elementor/editor/after_enqueue_scripts', function() {
            wp_enqueue_script('kingsway-elementor');
        });
    }

    /**
     * Admin notice for minimum PHP version
     */
    public function admin_notice_minimum_php_version() {
        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            esc_html__('"%1$s" requires PHP version %2$s or greater.', 'kingsway-plugin'),
            '<strong>' . esc_html__('Kingsway Plugin', 'kingsway-plugin') . '</strong>',
            '7.0'
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice for minimum Elementor version
     */
    public function admin_notice_minimum_elementor_version() {
        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            esc_html__('"%1$s" requires Elementor version %2$s or greater.', 'kingsway-plugin'),
            '<strong>' . esc_html__('Kingsway Plugin', 'kingsway-plugin') . '</strong>',
            '3.0.0'
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }
}