<?php
if (!defined('ABSPATH')) {
    exit;
}

class Kingsway_Video_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'kingsway_video';
    }

    public function get_title() {
        return esc_html__('Kingsway Video', 'kingsway-plugin');
    }

    public function get_icon() {
        return 'eicon-video-playlist';
    }

    public function get_categories() {
        return ['general'];
    }

    public function get_keywords() {
        return ['video', 'kingsway', 'player', 'widget'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Content', 'kingsway-plugin'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        // 内容类型选择
        $this->add_control(
            'content_type',
            [
                'label' => esc_html__('Content Type', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'video',
                'options' => [
                    'video' => esc_html__('Video', 'kingsway-plugin'),
                    'widget' => esc_html__('Widget', 'kingsway-plugin'),
                ],
            ]
        );

        // 隐藏字段存储选中的内容
        $this->add_control(
            'selected_id',
            [
                'label' => esc_html__('Selected ID', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => '',
            ]
        );

        $this->add_control(
            'selected_html',
            [
                'label' => esc_html__('Selected HTML', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => '',
            ]
        );

        $this->add_control(
            'selected_title',
            [
                'label' => esc_html__('Selected Title', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => '',
            ]
        );

        // 存储视频的宽高信息（用于计算aspect-ratio）
        $this->add_control(
            'video_width',
            [
                'label' => esc_html__('Video Width', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => '',
            ]
        );

        $this->add_control(
            'video_height',
            [
                'label' => esc_html__('Video Height', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => '',
            ]
        );

        // Open Library 按钮
        $this->add_control(
            'open_library_button',
            [
                'label' => esc_html__('Selected Content', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::BUTTON,
                'button_type' => 'default',
                'text' => esc_html__('Open Library', 'kingsway-plugin'),
                'event' => 'kingsway:open_library',
            ]
        );

        // 播放方式（仅单个视频）：直接播放 / 弹出播放
        $this->add_control(
            'embed_mode',
            [
                'label' => esc_html__('Embed Mode', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'inline',
                'options' => [
                    'inline' => esc_html__('Inline (直接播放)', 'kingsway-plugin'),
                    'popover' => esc_html__('Popover (弹出播放)', 'kingsway-plugin'),
                ],
                'condition' => [
                    'content_type' => 'video',
                ],
            ]
        );

        // Player Size（仅单个视频）：移动到内容标签页，放在Embed Mode下方
        $this->add_control(
            'player_size',
            [
                'label' => esc_html__('Player Size', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'responsive',
                'options' => [
                    'responsive' => esc_html__('Responsive', 'kingsway-plugin'),
                    'short-form' => esc_html__('Short Form', 'kingsway-plugin'),
                    'fixed' => esc_html__('Fixed Size', 'kingsway-plugin'),
                ],
                'condition' => [
                    'content_type' => 'video',
                ],
            ]
        );

        $this->add_control(
            'player_width',
            [
                'label' => esc_html__('Width', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 640,
                'condition' => [
                    'content_type' => 'video',
                    'player_size' => 'fixed',
                ],
            ]
        );

        $this->add_control(
            'player_height',
            [
                'label' => esc_html__('Height', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 360,
                'condition' => [
                    'content_type' => 'video',
                    'player_size' => 'fixed',
                ],
            ]
        );

        $this->end_controls_section();

        // Widget样式设置
        $this->start_controls_section(
            'widget_style_section',
            [
                'label' => esc_html__('Widget Style', 'kingsway-plugin'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'content_type' => 'widget',
                ],
            ]
        );

        $this->add_control(
            'widget_width',
            [
                'label' => esc_html__('Width', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '100%',
            ]
        );

        $this->add_control(
            'widget_height',
            [
                'label' => esc_html__('Height (optional)', 'kingsway-plugin'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => 'e.g. 360px',
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $content_type = $settings['content_type'];
        $selected_id = $settings['selected_id'];
        $selected_html = $settings['selected_html'];

        // 检查API key是否设置
        $api_key = get_option('kingsway-wordpress-plugin-api-key', '');
        if (empty($api_key)) {
            echo '<div class="elementor-alert elementor-alert-warning">' . 
                 esc_html__('Please set your Kingsway API Key in Settings > Kingsway.', 'kingsway-plugin') . 
                 '</div>';
            return;
        }

        echo '<div class="kingsway-video-container">';

        if (empty($selected_id)) {
            // 显示占位符
            echo '<div class="kingsway-placeholder" style="border: 2px dashed #ccc; padding: 40px; text-align: center; color: #999;">';
            echo '[ Kingsway Video ' . ($content_type === 'widget' ? 'Widget ' : '') . 'Placeholder ]';
            echo '</div>';
        } else {
            if (!empty($selected_html)) {
                // 如果是单个视频，根据「Embed Mode」和「Player Size」设置更新 HTML
                if ($content_type === 'video') {
                    // 处理 embed_mode
                    $is_popover = isset($settings['embed_mode']) && $settings['embed_mode'] === 'popover';
                    $selected_html = preg_replace(
                        '/data-popover-play="(true|false)"/',
                        'data-popover-play="' . ($is_popover ? 'true' : 'false') . '"',
                        $selected_html
                    );

                    // 处理 player_size
                    $player_size = isset($settings['player_size']) ? $settings['player_size'] : 'responsive';
                    $player_width = isset($settings['player_width']) ? intval($settings['player_width']) : 640;
                    $player_height = isset($settings['player_height']) ? intval($settings['player_height']) : 360;

                    // 提取现有的 style 属性（背景图片样式）
                    preg_match('/style="([^"]*)"/', $selected_html, $style_matches);
                    $image_style = $style_matches ? $style_matches[1] : '';
                    // 移除aspect-ratio、width、height相关的样式
                    $image_style = preg_replace('/aspect-ratio:\s*[^;]+;?/i', '', $image_style);
                    $image_style = preg_replace('/width:\s*[^;]+;?/i', '', $image_style);
                    $image_style = preg_replace('/height:\s*[^;]+;?/i', '', $image_style);
                    // 确保有背景图片样式
                    if (strpos($image_style, 'background-image') === false) {
                        $image_style = 'background-image: url(\'\'); background-size: auto 100%; background-position: center; background-repeat: no-repeat; background-color: #000;';
                    }

                    // 移除现有的尺寸相关属性
                    $selected_html = preg_replace('/\s+data-width="[^"]*"/', '', $selected_html);
                    $selected_html = preg_replace('/\s+data-height="[^"]*"/', '', $selected_html);
                    $selected_html = preg_replace('/\s+data-fixed-ratio="[^"]*"/', '', $selected_html);
                    $selected_html = preg_replace('/\s+style="[^"]*"/', '', $selected_html);

                    // 根据 player_size 添加新属性
                    if ($player_size === 'short-form') {
                        $selected_html = preg_replace(
                            '/(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/',
                            '$1 data-width="100%" data-fixed-ratio="16 / 9" style="aspect-ratio: 16 / 9;' . esc_attr($image_style) . '"$2',
                            $selected_html
                        );
                    } elseif ($player_size === 'fixed') {
                        $selected_html = preg_replace(
                            '/(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/',
                            '$1 data-width="' . esc_attr($player_width) . 'px" data-height="' . esc_attr($player_height) . 'px" style="width: ' . esc_attr($player_width) . 'px;' . esc_attr($image_style) . '"$2',
                            $selected_html
                        );
                    } else {
                        // responsive
                        $selected_html = preg_replace(
                            '/(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/',
                            '$1 data-width="100%" data-height="100%" style="aspect-ratio: 16 / 9;' . esc_attr($image_style) . '"$2',
                            $selected_html
                        );
                    }
                }

                echo $selected_html;
            }
        }

        echo '</div>';
    }

    protected function content_template() {
        ?>
        <div class="kingsway-video-container">
            <# if ( settings.selected_id ) { #>
                <#
                // 在编辑器预览模板中也应用 embed_mode 和 player_size 逻辑
                // 这样切换模式时预览会立即更新
                var html = settings.selected_html || '';
                if ( settings.content_type === 'video' && html ) {
                    // 处理 embed_mode
                    var isPopover = settings.embed_mode === 'popover';
                    html = html.replace(
                        /data-popover-play="(true|false)"/,
                        'data-popover-play="' + (isPopover ? 'true' : 'false') + '"'
                    );

                    // 处理 player_size
                    var playerSize = settings.player_size || 'responsive';
                    var playerWidth = settings.player_width || 640;
                    var playerHeight = settings.player_height || 360;

                    // 提取现有的 style 属性（背景图片样式）
                    var styleMatch = html.match(/style="([^"]*)"/);
                    var imageStyle = styleMatch ? styleMatch[1] : '';
                    // 移除aspect-ratio、width、height相关的样式
                    imageStyle = imageStyle.replace(/aspect-ratio:\s*[^;]+;?/gi, '');
                    imageStyle = imageStyle.replace(/width:\s*[^;]+;?/gi, '');
                    imageStyle = imageStyle.replace(/height:\s*[^;]+;?/gi, '');
                    // 确保有背景图片样式
                    if (imageStyle.indexOf('background-image') === -1) {
                        imageStyle = 'background-image: url(\'\'); background-size: auto 100%; background-position: center; background-repeat: no-repeat; background-color: #000;';
                    }

                    // 移除现有的尺寸相关属性
                    html = html.replace(/\s+data-width="[^"]*"/g, '');
                    html = html.replace(/\s+data-height="[^"]*"/g, '');
                    html = html.replace(/\s+data-fixed-ratio="[^"]*"/g, '');
                    html = html.replace(/\s+style="[^"]*"/g, '');

                    // 根据 player_size 添加新属性
                    if (playerSize === 'short-form') {
                        html = html.replace(
                            /(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/,
                            '$1 data-width="100%" data-fixed-ratio="16 / 9" style="aspect-ratio: 16 / 9;' + imageStyle + '"$2'
                        );
                    } else if (playerSize === 'fixed') {
                        html = html.replace(
                            /(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/,
                            '$1 data-width="' + playerWidth + 'px" data-height="' + playerHeight + 'px" style="width: ' + playerWidth + 'px;' + imageStyle + '"$2'
                        );
                    } else {
                        // responsive
                        html = html.replace(
                            /(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/,
                            '$1 data-width="100%" data-height="100%" style="aspect-ratio: 16 / 9;' + imageStyle + '"$2'
                        );
                    }
                }
                #>
                {{{ html }}}
            <# } else { #>
                <div class="kingsway-placeholder" style="border: 2px dashed #ccc; padding: 40px; text-align: center; color: #999;">
                    [ Kingsway Video <# if ( settings.content_type === 'widget' ) { #>Widget <# } #>Placeholder ]
                </div>
            <# } #>
        </div>
        <?php
    }
}
