/**
 * Kingsway Elementor Integration
 * 使用共用的 KingswayLibrary 模块
 */
(function ($) {
  "use strict";

  const POLL_INTERVAL = 100;
  const MAX_RETRIES = 50; // 最多等待5秒

  // 等待 KingswayLibrary 加载
  function waitForLibrary(callback, retries = 0) {
    if (typeof KingswayLibrary !== "undefined") {
      callback();
    } else if (retries < MAX_RETRIES) {
      setTimeout(() => waitForLibrary(callback, retries + 1), POLL_INTERVAL);
    } else {
      console.error("KingswayLibrary failed to load after timeout");
    }
  }

  // 初始化库配置
  function initLibrary() {
    if (
      typeof kingswayPlugin !== "undefined" &&
      typeof KingswayLibrary !== "undefined"
    ) {
      KingswayLibrary.init({
        apiRoot: kingswayPlugin.apiRoot,
        nonce: kingswayPlugin.nonce,
        env: kingswayPlugin.env || "production",
      });
    }
  }

  waitForLibrary(function () {
    initLibrary();

    // 辅助函数：根据 embed_mode 更新 HTML 中的 data-popover-play
    function applyEmbedModeToHtml(html, embedMode) {
      if (!html) return html;
      const isPopover = embedMode === "popover";
      if (html.includes("data-popover-play=")) {
        return html.replace(
          /data-popover-play="(true|false)"/,
          `data-popover-play="${isPopover ? "true" : "false"}"`
        );
      } else {
        // 如果没有 data-popover-play 属性，添加它
        return html.replace(
          /(<div[^>]*class="from-kingsway-plugin"[^>]*)(>)/,
          `$1 data-popover-play="${isPopover ? "true" : "false"}"$2`
        );
      }
    }

    // 辅助函数：解析 HTML 中的 player size
    function parsePlayerSizeFromHtml(html) {
      if (!html) return "responsive";

      // 检查是否有 data-fixed-ratio（short-form）
      if (html.includes("data-fixed-ratio=")) {
        return "short-form";
      }

      // 检查是否是 fixed size（有明确的 px 值）
      const widthMatch = html.match(/data-width="(\d+)px"/);
      const heightMatch = html.match(/data-height="(\d+)px"/);
      if (widthMatch && heightMatch) {
        return "fixed";
      }

      // 默认是 responsive
      return "responsive";
    }

    // 辅助函数：解析 HTML 中的 player width 和 height
    function parsePlayerDimensionsFromHtml(html) {
      const widthMatch = html.match(/data-width="(\d+)px"/);
      const heightMatch = html.match(/data-height="(\d+)px"/);
      return {
        width: widthMatch ? parseInt(widthMatch[1], 10) : 640,
        height: heightMatch ? parseInt(heightMatch[1], 10) : 360,
      };
    }

    // 辅助函数：计算最大公约数
    function gcd(a, b) {
      return b === 0 ? a : gcd(b, a % b);
    }

    // 辅助函数：根据 player size 更新 HTML 中的尺寸属性
    function applyPlayerSizeToHtml(
      html,
      playerSize,
      playerWidth,
      playerHeight,
      videoWidth,
      videoHeight
    ) {
      if (!html) return html;

      // 获取 div 标签中的 style 属性，提取背景图片样式
      const divMatch = html.match(
        /<div[^>]*class="from-kingsway-plugin"[^>]*>/
      );
      if (!divMatch) return html;

      const divTag = divMatch[0];
      const styleMatch = divTag.match(/style="([^"]*)"/);
      // 从style中提取背景图片相关的样式，移除尺寸相关的样式
      let imageStyle = styleMatch ? styleMatch[1] : "";
      // 移除aspect-ratio、width、height相关的样式
      imageStyle = imageStyle.replace(/aspect-ratio:\s*[^;]+;?/g, "");
      imageStyle = imageStyle.replace(/width:\s*[^;]+;?/g, "");
      imageStyle = imageStyle.replace(/height:\s*[^;]+;?/g, "");
      // 确保有背景图片样式
      if (!imageStyle.includes("background-image")) {
        imageStyle =
          "background-image: url(''); background-size: auto 100%; background-position: center; background-repeat: no-repeat; background-color: #000;";
      }

      // 计算视频的宽高比（用于Responsive模式）
      let aspectRatio = "16 / 9"; // 默认值
      if (videoWidth && videoHeight) {
        const vWidth = parseInt(videoWidth, 10) || 16;
        const vHeight = parseInt(videoHeight, 10) || 9;
        const commonDivisor = gcd(vWidth, vHeight);
        aspectRatio = `${vWidth / commonDivisor} / ${vHeight / commonDivisor}`;
      }

      // 构建新的属性字符串
      let newAttrs = [];
      if (playerSize === "short-form") {
        newAttrs.push('data-width="100%"');
        newAttrs.push('data-fixed-ratio="16 / 9"');
        newAttrs.push(`style="aspect-ratio: 16 / 9;${imageStyle}"`);
      } else if (playerSize === "fixed") {
        // Fixed Size: 不使用aspect-ratio，但在style中添加width
        newAttrs.push(`data-width="${playerWidth}px"`);
        newAttrs.push(`data-height="${playerHeight}px"`);
        newAttrs.push(`style="width: ${playerWidth}px;${imageStyle}"`);
      } else {
        // Responsive: 使用视频的实际宽高比
        newAttrs.push('data-width="100%"');
        newAttrs.push('data-height="100%"');
        newAttrs.push(`style="aspect-ratio: ${aspectRatio};${imageStyle}"`);
      }

      // 构建新的 div 标签
      // 保留除了尺寸相关的其他属性
      let newDivTag = divTag
        .replace(/\s+data-width="[^"]*"/g, "")
        .replace(/\s+data-height="[^"]*"/g, "")
        .replace(/\s+data-fixed-ratio="[^"]*"/g, "")
        .replace(/\s+style="[^"]*"/g, "");

      // 确保newDivTag末尾有一个空格，然后添加新属性
      newDivTag = newDivTag.trim();
      newDivTag = newDivTag.replace(/>$/, " " + newAttrs.join(" ") + ">");

      // 替换整个 HTML 中的 div 标签
      return html.replace(
        /<div[^>]*class="from-kingsway-plugin"[^>]*>/,
        newDivTag
      );
    }

    // 打开库
    function openLibrary(widgetId) {
      const container = window.elementor?.getContainer(widgetId);
      if (!container) {
        console.error("Cannot find Elementor container");
        return;
      }

      const settings = container.settings.attributes;
      const contentType = settings.content_type || "video";
      const DEFAULT_PLAYER_SIZE = "responsive";
      const DEFAULT_PLAYER_WIDTH = 640;
      const DEFAULT_PLAYER_HEIGHT = 360;
      const DEFAULT_EMBED_MODE = "inline";
      const DEFAULT_WIDGET_WIDTH = "100%";
      const DEFAULT_WIDGET_HEIGHT = "";

      // 监听 embed_mode 变化，实时更新 selected_html
      // 这样切换模式时，相当于用当前模式重新嵌入同一个视频
      if (
        !container.__kingswayEmbedModeListenerBound &&
        container.settings?.on
      ) {
        container.__kingswayEmbedModeListenerBound = true;

        container.settings.on("change:embed_mode", function () {
          const s = container.settings.attributes || {};
          // 只处理单个视频的情况
          if ((s.content_type || "video") !== "video") return;
          if (!s.selected_html) return;

          const currentMode = s.embed_mode || DEFAULT_EMBED_MODE;

          // 1）更新用于保存的 HTML（selected_html）
          const updatedHtml = applyEmbedModeToHtml(
            s.selected_html,
            currentMode
          );
          container.settings.set({ selected_html: updatedHtml });

          // 2）实时预览：仅更新 DOM 上的 data-popover-play 属性，避免破坏 SDK 已有实例
          const isPopover = currentMode === "popover";
          const widgetSelector = `.elementor-element-${widgetId} .kingsway-video-container`;
          const previewEl = document.querySelector(widgetSelector);
          if (previewEl) {
            const playerEl = previewEl.querySelector(".from-kingsway-plugin");
            if (playerEl) {
              playerEl.setAttribute(
                "data-popover-play",
                isPopover ? "true" : "false"
              );
            }
          }

          // 标记编辑器已更改
          if (elementor && elementor.saver) {
            elementor.saver.setFlagEditorChange(true);
          }
        });
      }

      // 监听 player_size 变化，实时更新 selected_html
      if (
        !container.__kingswayPlayerSizeListenerBound &&
        container.settings?.on
      ) {
        container.__kingswayPlayerSizeListenerBound = true;

        container.settings.on("change:player_size", function () {
          const s = container.settings.attributes || {};
          // 只处理单个视频的情况
          if ((s.content_type || "video") !== "video") return;
          if (!s.selected_html) return;

          const currentSize = s.player_size || DEFAULT_PLAYER_SIZE;
          const currentWidth = s.player_width || DEFAULT_PLAYER_WIDTH;
          const currentHeight = s.player_height || DEFAULT_PLAYER_HEIGHT;
          const videoWidth = s.video_width || 16;
          const videoHeight = s.video_height || 9;

          // 更新用于保存的 HTML（selected_html）
          let updatedHtml = applyPlayerSizeToHtml(
            s.selected_html,
            currentSize,
            currentWidth,
            currentHeight,
            videoWidth,
            videoHeight
          );
          // 保持 embed_mode
          updatedHtml = applyEmbedModeToHtml(
            updatedHtml,
            s.embed_mode || DEFAULT_EMBED_MODE
          );
          container.settings.set({ selected_html: updatedHtml });

          // 实时预览：重新渲染整个 HTML
          const widgetSelector = `.elementor-element-${widgetId} .kingsway-video-container`;
          const previewEl = document.querySelector(widgetSelector);
          if (previewEl) {
            previewEl.innerHTML = updatedHtml;
            // 重新加载脚本
            requestAnimationFrame(() => {
              if (typeof KingswayLibrary !== "undefined") {
                KingswayLibrary.loadPlayerScript("video");
              }
            });
          }

          // 标记编辑器已更改
          if (elementor && elementor.saver) {
            elementor.saver.setFlagEditorChange(true);
          }
        });
      }

      // 监听 player_width 和 player_height 变化，使用防抖（1秒）
      if (
        !container.__kingswayPlayerDimensionsListenerBound &&
        container.settings?.on
      ) {
        container.__kingswayPlayerDimensionsListenerBound = true;
        let debounceTimer = null;

        const updatePlayerDimensions = function () {
          const s = container.settings.attributes || {};
          // 只处理单个视频且为 fixed size 的情况
          if ((s.content_type || "video") !== "video") return;
          if ((s.player_size || DEFAULT_PLAYER_SIZE) !== "fixed") return;
          if (!s.selected_html) return;

          const currentSize = s.player_size || DEFAULT_PLAYER_SIZE;
          const currentWidth = s.player_width || DEFAULT_PLAYER_WIDTH;
          const currentHeight = s.player_height || DEFAULT_PLAYER_HEIGHT;
          const videoWidth = s.video_width || 16;
          const videoHeight = s.video_height || 9;

          // 更新用于保存的 HTML（selected_html）
          let updatedHtml = applyPlayerSizeToHtml(
            s.selected_html,
            currentSize,
            currentWidth,
            currentHeight,
            videoWidth,
            videoHeight
          );
          // 保持 embed_mode
          updatedHtml = applyEmbedModeToHtml(
            updatedHtml,
            s.embed_mode || DEFAULT_EMBED_MODE
          );
          container.settings.set({ selected_html: updatedHtml });

          // 实时预览：重新渲染整个 HTML
          const widgetSelector = `.elementor-element-${widgetId} .kingsway-video-container`;
          const previewEl = document.querySelector(widgetSelector);
          if (previewEl) {
            previewEl.innerHTML = updatedHtml;
            // 重新加载脚本
            requestAnimationFrame(() => {
              if (typeof KingswayLibrary !== "undefined") {
                KingswayLibrary.loadPlayerScript("video");
              }
            });
          }

          // 标记编辑器已更改
          if (elementor && elementor.saver) {
            elementor.saver.setFlagEditorChange(true);
          }
        };

        container.settings.on("change:player_width", function () {
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(updatePlayerDimensions, 1000);
        });

        container.settings.on("change:player_height", function () {
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(updatePlayerDimensions, 1000);
        });
      }

      KingswayLibrary.open({
        contentType: contentType,
        videoOptions: {
          playerSize: settings.player_size || DEFAULT_PLAYER_SIZE,
          playerWidth: settings.player_width || DEFAULT_PLAYER_WIDTH,
          playerHeight: settings.player_height || DEFAULT_PLAYER_HEIGHT,
          embedMode: settings.embed_mode || DEFAULT_EMBED_MODE,
        },
        widgetOptions: {
          widgetWidth: settings.widget_width || DEFAULT_WIDGET_WIDTH,
          widgetHeight: settings.widget_height || DEFAULT_WIDGET_HEIGHT,
        },
        onSelect: (type, item, html) => {
          // 如果是单个视频，确保 HTML 中的 embed_mode 与当前设置一致
          if (type === "video") {
            html = applyEmbedModeToHtml(
              html,
              settings.embed_mode || DEFAULT_EMBED_MODE
            );

            // 保存视频的宽高信息（用于计算aspect-ratio）
            const videoWidth =
              item.resolutions && item.resolutions[0]
                ? item.resolutions[0].width
                : 16;
            const videoHeight =
              item.resolutions && item.resolutions[0]
                ? item.resolutions[0].height
                : 9;

            // 更新控件设置（会触发 Elementor 重新渲染预览区域）
            container.settings.set({
              selected_id: String(item.id),
              selected_html: html,
              selected_title: item.title || "",
              video_width: videoWidth,
              video_height: videoHeight,
            });
          } else {
            // 更新控件设置（会触发 Elementor 重新渲染预览区域）
            container.settings.set({
              selected_id: String(item.extendedId),
              selected_html: html,
              selected_title: item.classify?.classifyName || "",
            });
          }

          // 标记编辑器已更改，避免离开页面时不提示
          if (elementor && elementor.saver) {
            elementor.saver.setFlagEditorChange(true);
          }

          // 直接更新当前小部件的预览 DOM，做到即时可见
          const widgetSelector = `.elementor-element-${widgetId} .kingsway-video-container`;
          const previewEl = document.querySelector(widgetSelector);
          if (previewEl) {
            previewEl.innerHTML = html;

            // innerHTML 插入的 <script> 标签不会被执行，需要手动加载 SDK
            // 使用 requestAnimationFrame 确保 DOM 已更新，比 setTimeout 更可靠
            const scriptType = type === "video" ? "video" : "widget";
            requestAnimationFrame(() => {
              if (typeof KingswayLibrary !== "undefined") {
                KingswayLibrary.loadPlayerScript(scriptType);
              }
            });
          }
        },
      });
    }

    // 监听 Elementor 编辑器事件
    $(window).on("elementor/frontend/init", function () {
      if (window.elementor) {
        // 编辑器中点击「Open Library」按钮时打开库
        elementor.channels.editor.on("kingsway:open_library", function () {
          const panel = elementor.getPanelView()?.getCurrentPageView();
          if (panel) {
            const model = panel
              .getOption("editedElementView")
              ?.getContainer()?.model;
            const widgetId = model?.get("id");
            if (widgetId) {
              openLibrary(widgetId);
            }
          }
        });
      }
    });
  });
})(jQuery);
