/**
 * Kingsway Library - 共用弹窗模块
 * 供 Gutenberg、Elementor、WooCommerce 等环境使用
 */
(function (root, factory) {
  if (typeof define === "function" && define.amd) {
    define([], factory);
  } else if (typeof module === "object" && module.exports) {
    module.exports = factory();
  } else {
    root.KingswayLibrary = factory();
  }
})(typeof self !== "undefined" ? self : this, function () {
  "use strict";

  // ============ 常量配置 ============
  const CONSTANTS = {
    PAGE_SIZE: {
      VIDEO: 8,
      WIDGET: 10,
    },
    DEFAULT_PLAYER: {
      WIDTH: 640,
      HEIGHT: 360,
      SIZE: "responsive",
      EMBED_MODE: "inline",
    },
    DEFAULT_WIDGET: {
      WIDTH: "100%",
      HEIGHT: "",
    },
    ENV_MAP: {
      production: { prefix: "", widget: "prod" },
      development: { prefix: "test-", widget: "dev" },
      preRelease: { prefix: "pre-", widget: "pre" },
    },
    SDK_URLS: {
      player: {
        production:
          "https://websdk.kingswayvideo.com/vod-player/latest/vod-player.min.js",
        development:
          "https://test-v-assets.kingswayvideo.com/vod-player/dev/vod-player.min.js",
        preRelease:
          "https://pre-v-assets.kingswayvideo.com/vod-player/latest/vod-player.min.js",
      },
      widget: {
        production:
          "https://websdk.kingswayvideo.com/video-widgets/latest/video-widgets.min.js",
        development:
          "https://test-v-assets.kingswayvideo.com/video-widgets/dev/video-widgets.min.js",
        preRelease:
          "https://pre-v-assets.kingswayvideo.com/video-widgets/latest/video-widgets.min.js",
      },
    },
  };

  // ============ 配置模块 ============
  const Config = {
    apiRoot: "",
    nonce: "",
    env: "production",
    apiKey: null,
    sdkUrl: "",
    apiHost: "",

    init(options) {
      if (options.apiRoot) this.apiRoot = options.apiRoot;
      if (options.nonce) this.nonce = options.nonce;
      if (options.env) this.setEnv(options.env);
    },

    setEnv(newEnv) {
      this.env = newEnv;
      const envConfig =
        CONSTANTS.ENV_MAP[newEnv] || CONSTANTS.ENV_MAP.production;

      this.sdkUrl =
        CONSTANTS.SDK_URLS.player[newEnv] ||
        CONSTANTS.SDK_URLS.player.production;
      this.apiHost = `https://${envConfig.prefix}api.kingswayvideo.com`;
    },

    getEnv() {
      return this.env;
    },

    getSdkUrl() {
      return this.sdkUrl;
    },

    getWidgetEnvValue() {
      const envConfig =
        CONSTANTS.ENV_MAP[this.env] || CONSTANTS.ENV_MAP.production;
      return envConfig.widget;
    },

    getWidgetSdkUrl() {
      const env = this.env;
      if (env === "development") {
        return CONSTANTS.SDK_URLS.widget.development;
      } else if (env === "preRelease") {
        return CONSTANTS.SDK_URLS.widget.preRelease;
      }
      return CONSTANTS.SDK_URLS.widget.production;
    },

    clearApiKey() {
      this.apiKey = null;
    },
  };

  // ============ API 模块 ============
  const Api = {
    async getApiKey() {
      if (Config.apiKey) return Config.apiKey;

      if (!Config.apiRoot) {
        console.error("Kingsway: apiRoot not configured");
        return null;
      }

      try {
        const response = await fetch(Config.apiRoot + "api-key/", {
          headers: { "X-WP-Nonce": Config.nonce },
        });
        const data = await response.json();
        Config.apiKey = data.api_key || null;
        return Config.apiKey;
      } catch (error) {
        console.error("Error fetching API key:", error);
        return null;
      }
    },

    async fetchVideoList(page = 1, folderId = null, keyword = "") {
      const key = await this.getApiKey();
      if (!key) return { list: [], total: 0 };

      try {
        const params = new URLSearchParams({
          page,
          size: CONSTANTS.PAGE_SIZE.VIDEO,
        });
        if (folderId) params.append("folderId", folderId);
        if (keyword) params.append("keyword", keyword);

        const response = await fetch(
          `${
            Config.apiHost
          }/vod/video/get-custom-video-list-by-folder?needPureList=true&${params.toString()}`,
          {
            headers: { Authorization: `Bearer ${key}` },
          }
        );
        const data = await response.json();

        // 过滤掉 project 类型
        if (data.data?.list) {
          data.data.list = data.data.list.filter((v) => v.type !== "project");
        }

        return data.data || { list: [], total: 0 };
      } catch (error) {
        console.error("Error fetching video list:", error);
        return { list: [], total: 0 };
      }
    },

    async fetchWidgetList(page = 1) {
      const key = await this.getApiKey();
      if (!key) return { list: [], total: 0 };

      try {
        const params = new URLSearchParams({
          page,
          size: CONSTANTS.PAGE_SIZE.WIDGET,
        });

        const response = await fetch(
          `${Config.apiHost}/video-list-extended/list?${params.toString()}`,
          {
            headers: { Authorization: `Bearer ${key}` },
          }
        );
        const data = await response.json();

        if (data.status === "success" && data.data) {
          return data.data;
        }
        return { list: [], total: 0 };
      } catch (error) {
        console.error("Error fetching video widgets:", error);
        return { list: [], total: 0 };
      }
    },
  };

  // ============ 工具函数模块 ============
  const Utils = {
    formatDuration(duration) {
      const hours = Math.floor(duration / 3600);
      const minutes = Math.floor((duration % 3600) / 60);
      const seconds = Math.floor(duration % 60);

      if (hours > 0) {
        return `${String(hours).padStart(2, "0")}:${String(minutes).padStart(
          2,
          "0"
        )}:${String(seconds).padStart(2, "0")}`;
      }
      return `${String(minutes).padStart(2, "0")}:${String(seconds).padStart(
        2,
        "0"
      )}`;
    },

    formatDate(timestamp) {
      return new Date(timestamp * 1000).toLocaleDateString();
    },

    createScriptTag(src) {
      return `<script>
var p_tag = document.createElement('script');
p_tag.type = 'text/javascript';
p_tag.defer = true;
p_tag.charset = 'utf-8';
p_tag.src = '${src}';
document.getElementsByTagName('head')[0].appendChild(p_tag);
</script>`;
    },

    getThumbnailUrl(video) {
      return video.player?.thumbnailUrl || video.cover || "";
    },

    getPlayIconUrl(thumbnailUrl) {
      if (!thumbnailUrl) return "";
      return thumbnailUrl.replace(
        "/cover.jpg",
        "/cover.jpg?imageMogr2/format/webp/thumbnail/!30p"
      );
    },
  };

  // 数据存储：用内存 Map 存引用，避免 JSON.parse/序列化污染
  const DataStore = {
    seq: 0,
    videos: new Map(),
    widgets: new Map(),
    folders: new Map(),
    nextId(prefix) {
      this.seq += 1;
      return `${prefix}-${this.seq}`;
    },
    reset(type) {
      if (type === "video") {
        this.videos.clear();
        this.folders.clear();
      } else if (type === "widget") {
        this.widgets.clear();
      }
    },
  };

  // ============ 工具函数：计算最大公约数 ============
  function gcd(a, b) {
    return b === 0 ? a : gcd(b, a % b);
  }

  // ============ HTML 生成模块 ============
  const HtmlGenerator = {
    generateVideoHtml(video, options = {}) {
      const {
        playerSize = CONSTANTS.DEFAULT_PLAYER.SIZE,
        playerWidth = CONSTANTS.DEFAULT_PLAYER.WIDTH,
        playerHeight = CONSTANTS.DEFAULT_PLAYER.HEIGHT,
        embedMode = CONSTANTS.DEFAULT_PLAYER.EMBED_MODE,
      } = options;

      const thumbnailUrl = Utils.getThumbnailUrl(video);
      const playIconUrl = Utils.getPlayIconUrl(thumbnailUrl);
      const imageStyle = `background-image: url('${playIconUrl}'); background-size: auto 100%; background-position: center; background-repeat: no-repeat; background-color: #000;`;

      // 从视频数据中获取宽高（用于计算aspect-ratio）
      const videoWidth =
        video.resolutions && video.resolutions[0]
          ? video.resolutions[0].width
          : 16;
      const videoHeight =
        video.resolutions && video.resolutions[0]
          ? video.resolutions[0].height
          : 9;
      const commonDivisor = gcd(videoWidth, videoHeight);
      const aspectRatio = `${videoWidth / commonDivisor} / ${
        videoHeight / commonDivisor
      }`;

      let sizeStr = "";
      if (playerSize === "short-form") {
        sizeStr = `data-width="100%" data-fixed-ratio="16 / 9" style="aspect-ratio: 16 / 9;${imageStyle}"`;
      } else if (playerSize === "fixed") {
        // Fixed Size: 不使用aspect-ratio，但在style中添加width
        sizeStr = `data-width="${playerWidth}px" data-height="${playerHeight}px" style="width: ${playerWidth}px;${imageStyle}"`;
      } else {
        // Responsive: 使用视频的实际宽高比
        sizeStr = `data-width="100%" data-height="100%" style="aspect-ratio: ${aspectRatio};${imageStyle}"`;
      }

      const envAttr =
        Config.env !== "production" ? `data-env="${Config.env}"` : "";
      const popoverAttr = `data-popover-play="${
        embedMode === "popover" ? "true" : "false"
      }"`;

      return `<div class="from-kingsway-plugin" data-kingsway-player="${
        video.id
      }" ${envAttr} ${sizeStr} ${popoverAttr}></div>
${Utils.createScriptTag(Config.sdkUrl)}`;
    },

    generateWidgetHtml(widget, options = {}) {
      const {
        widgetWidth = CONSTANTS.DEFAULT_WIDGET.WIDTH,
        widgetHeight = CONSTANTS.DEFAULT_WIDGET.HEIGHT,
      } = options;

      const envValue = Config.getWidgetEnvValue();
      const widgetJsUrl = Config.getWidgetSdkUrl();

      let html = `<div data-kingsway-video-list-extend="${widget.extendedId}" data-video-list-extend-type="${widget.type}" data-video-list-extend-env="${envValue}" data-width="${widgetWidth}"`;

      if (widgetHeight) {
        html += ` data-height="${widgetHeight}"`;
      }

      html += `></div>${Utils.createScriptTag(widgetJsUrl)}`;
      return html;
    },
  };

  // ============ 脚本加载模块 ============
  const ScriptLoader = {
    loadPlayerScript(type = "video") {
      // 确保配置已初始化（如果还没有初始化）
      if (
        !Config.sdkUrl &&
        typeof window !== "undefined" &&
        window.kingswayPlugin
      ) {
        Config.init({
          apiRoot: window.kingswayPlugin.apiRoot || "",
          nonce: window.kingswayPlugin.nonce || "",
          env: window.kingswayPlugin.env || "production",
        });
      }

      const script = document.createElement("script");

      if (type === "widget") {
        window.__kingsway_need_single_init = false;
        script.src = Config.getWidgetSdkUrl();
      } else {
        script.src = Config.sdkUrl || CONSTANTS.SDK_URLS.player.production;
      }

      script.type = "text/javascript";
      script.defer = true;
      script.charset = "utf-8";
      document.head.appendChild(script);
    },
  };

  // ============ 样式注入模块 ============
  const StyleInjector = {
    injectStyles() {
      // 加载共享样式
      if (!document.getElementById("kingsway-shared")) {
        const link = document.createElement("link");
        link.id = "kingsway-shared";
        link.rel = "stylesheet";
        link.href =
          window.kingswayPlugin?.cssUrl ||
          "/wp-content/plugins/kingsway-wordpress-plugin/assets/css/kingsway-shared.css";
        document.head.appendChild(link);
      }

      // 注入弹窗特定样式
      if (document.getElementById("kingsway-library-styles")) return;

      const style = document.createElement("style");
      style.id = "kingsway-library-styles";
      style.textContent = this.getStyles();
      document.head.appendChild(style);
    },

    getStyles() {
      return `
      .kingsway-library-modal {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        z-index: 999999;
        display: flex;
        align-items: center;
        justify-content: center;
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
      }
      .kingsway-library-modal__overlay {
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0,0,0,0.6);
      }
      .kingsway-library-modal__content {
        position: relative;
        background: #fff;
        width: 90%;
        max-width: 1100px;
        max-height: 85vh;
        border-radius: 8px;
        display: flex;
        flex-direction: column;
        box-shadow: 0 10px 40px rgba(0,0,0,0.3);
      }
      .kingsway-library-modal__header {
        display: flex;
        align-items: center;
        padding: 16px 20px;
        border-bottom: 1px solid #eee;
        flex-shrink: 0;
      }
      .kingsway-library-modal__back {
        background: none;
        border: none;
        color: #0073aa;
        cursor: pointer;
        margin-right: 12px;
        font-size: 14px;
        padding: 4px 8px;
      }
      .kingsway-library-modal__back:hover {
        text-decoration: underline;
      }
      .kingsway-library-modal__title {
        flex: 1;
        margin: 0;
        font-size: 18px;
        font-weight: 600;
      }
      .kingsway-library-modal__close {
        background: none;
        border: none;
        font-size: 28px;
        cursor: pointer;
        color: #666;
        line-height: 1;
        padding: 0 4px;
      }
      .kingsway-library-modal__close:hover {
        color: #333;
      }
      .kingsway-library-modal__body {
        padding: 20px;
        overflow-y: auto;
        flex: 1;
        position: relative;
      }
      .kingsway-library-modal__loading {
        text-align: center;
        padding: 60px;
        color: #666;
        font-size: 16px;
      }

      .kingsway-library-modal__loading-overlay {
        position: absolute;
        inset: 0;
        display: flex;
        align-items: center;
        justify-content: center;
        background: rgba(255, 255, 255, 0.75);
        z-index: 10;
      }
      .kingsway-library-modal__loading-overlay .kingsway-library-modal__loading {
        padding: 0;
      }
      .kingsway-library-modal__body--locked {
        overflow-y: hidden;
      }
      
      .kingsway-library-no-key {
        text-align: center;
        padding: 40px 20px;
      }
      .kingsway-library-no-key__icon {
        font-size: 48px;
        margin-bottom: 16px;
      }
      .kingsway-library-no-key h3 {
        margin: 0 0 12px;
        font-size: 20px;
        color: #333;
      }
      .kingsway-library-no-key p {
        color: #666;
        margin-bottom: 24px;
      }
      .kingsway-library-no-key__actions {
        display: flex;
        justify-content: center;
        gap: 12px;
      }
      
      .kingsway-library-select-type {
        padding: 20px;
      }
      .kingsway-library-select-type h3 {
        margin: 0 0 8px;
        font-size: 18px;
        color: #333;
      }
      .kingsway-library-select-type p {
        color: #666;
        margin-bottom: 24px;
      }
      .kingsway-library-select-type__options {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 20px;
        max-width: 600px;
        margin: 0 auto;
      }
      .kingsway-library-select-type__option {
        border: 2px solid #e0e0e0;
        border-radius: 12px;
        padding: 30px 20px;
        text-align: center;
        cursor: pointer;
        transition: all 0.2s;
      }
      .kingsway-library-select-type__option:hover {
        border-color: #0073aa;
        background-color: #f8fbfd;
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 115, 170, 0.15);
      }
      .kingsway-library-select-type__option-icon {
        font-size: 48px;
        margin-bottom: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: #333;
      }
      .kingsway-library-select-type__option-icon svg {
        width: 48px;
        height: 48px;
      }
      .kingsway-library-select-type__option-title {
        font-size: 18px;
        font-weight: 600;
        color: #333;
        margin-bottom: 8px;
      }
      .kingsway-library-select-type__option-desc {
        font-size: 14px;
        color: #666;
      }
      
      .kingsway-library-toolbar {
        display: flex;
        flex-wrap: wrap;
        gap: 16px;
        margin-bottom: 20px;
      }
      .kingsway-library-toolbar__search {
        display: flex;
        gap: 8px;
      }
      .kingsway-library-toolbar__search input {
        padding: 8px 12px;
        border: 1px solid #ddd;
        border-radius: 4px;
        font-size: 14px;
        width: 200px;
      }
      .kingsway-library-toolbar__group {
        display: flex;
        align-items: center;
        gap: 8px;
      }
      .kingsway-library-toolbar__label {
        font-size: 13px;
        color: #666;
      }
      .kingsway-library-toolbar__group label {
        display: flex;
        align-items: center;
        gap: 4px;
        font-size: 13px;
        cursor: pointer;
      }
      .kingsway-library-toolbar__group input[type="text"] {
        width: 60px;
        padding: 4px 8px;
        border: 1px solid #ddd;
        border-radius: 4px;
        font-size: 13px;
      }
      
      .kingsway-library-breadcrumb {
        display: flex;
        align-items: center;
        margin-bottom: 16px;
        font-size: 14px;
      }
      .kingsway-library-breadcrumb__link {
        color: #0073aa;
        cursor: pointer;
        text-decoration: underline;
      }
      .kingsway-library-breadcrumb__link:hover {
        color: #005177;
      }
      .kingsway-library-breadcrumb__sep {
        margin: 0 8px;
        color: #999;
      }
      .kingsway-library-breadcrumb__current {
        color: #333;
      }
      
      .kingsway-library-grid {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 20px;
        padding: 0;
      }
      @media (max-width: 900px) {
        .kingsway-library-grid {
          grid-template-columns: repeat(3, 1fr);
        }
      }
      @media (max-width: 600px) {
        .kingsway-library-grid {
          grid-template-columns: repeat(2, 1fr);
        }
      }
      
      .kingsway-library-pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 12px;
        margin-top: 24px;
        padding-top: 16px;
        border-top: 1px solid #eee;
      }
      .kingsway-library-pagination__info {
        font-size: 14px;
        color: #666;
      }
      
      .kingsway-library-btn {
        padding: 8px 16px;
        border: 1px solid #ddd;
        background: #fff;
        border-radius: 4px;
        cursor: pointer;
        font-size: 14px;
        transition: all 0.2s;
        color: black;
      }
      .kingsway-library-btn:hover:not(:disabled) {
        border-color: #0073aa;
        color: #0073aa;
      }
      .kingsway-library-btn:disabled {
        opacity: 0.5;
        cursor: not-allowed;
      }
      .kingsway-library-btn--primary {
        background: #0073aa;
        color: #fff;
        border-color: #0073aa;
      }
      .kingsway-library-btn--primary:hover:not(:disabled) {
        background: #005177;
        border-color: #005177;
        color: #fff;
      }
      
      .kingsway-library-empty {
        grid-column: 1 / -1;
        text-align: center;
        padding: 40px;
        color: #999;
      }
      
      .kingsway-library-modal .c-folder-card,
      .kingsway-library-modal .c-video-card {
        margin: 0;
        width: 100%;
      }
    `;
    },
  };

  // ============ UI 渲染模块 ============
  const UIRenderer = {
    setBodyLoading(body, isLoading) {
      if (!body) return;

      const existingOverlay = body.querySelector(
        ".kingsway-library-modal__loading-overlay"
      );

      if (isLoading) {
        // Lock current height so modal doesn't "shrink" during loading.
        // Only lock if we have existing content height to preserve.
        const rect = body.getBoundingClientRect();
        const currentHeight = Math.max(0, Math.round(rect.height || 0));
        if (currentHeight > 0) {
          body.style.minHeight = `${currentHeight}px`;
        }
        body.classList.add("kingsway-library-modal__body--locked");

        if (!existingOverlay) {
          const overlay = document.createElement("div");
          overlay.className = "kingsway-library-modal__loading-overlay";
          overlay.innerHTML =
            '<div class="kingsway-library-modal__loading">Loading...</div>';
          body.appendChild(overlay);
        }
        return;
      }

      // Unlock + remove overlay
      body.classList.remove("kingsway-library-modal__body--locked");
      body.style.minHeight = "";
      if (existingOverlay) existingOverlay.remove();
    },

    createModal() {
      StyleInjector.injectStyles();

      const existingModal = document.querySelector(".kingsway-library-modal");
      if (existingModal) existingModal.remove();

      const modal = document.createElement("div");
      modal.className = "kingsway-library-modal";
      modal.innerHTML = `
        <div class="kingsway-library-modal__overlay"></div>
        <div class="kingsway-library-modal__content">
          <div class="kingsway-library-modal__header">
            <button class="kingsway-library-modal__back" style="display: none;">← Back</button>
            <h3 class="kingsway-library-modal__title">Kingsway Library</h3>
            <button class="kingsway-library-modal__close">×</button>
          </div>
          <div class="kingsway-library-modal__body">
            <div class="kingsway-library-modal__loading">Loading...</div>
          </div>
        </div>
      `;
      document.body.appendChild(modal);

      return modal;
    },

    renderNoApiKey(modal, onRefresh) {
      const body = modal.querySelector(".kingsway-library-modal__body");
      body.innerHTML = `
        <div class="kingsway-library-no-key">
          <div class="kingsway-library-no-key__icon">⚠️</div>
          <h3>API Key Not Set</h3>
          <p>Please set your Kingsway API Key in WordPress Settings first.</p>
          <div class="kingsway-library-no-key__actions">
            <a href="${window.location.origin}/wp-admin/options-general.php?page=kingsway-settings" target="_blank" class="kingsway-library-btn kingsway-library-btn--primary">Go to Settings</a>
            <button class="kingsway-library-btn" data-action="refresh">Refresh</button>
          </div>
        </div>
      `;
      body.querySelector('[data-action="refresh"]').onclick = onRefresh;
    },

    renderSelectType(modal, onSelectVideo, onSelectWidget) {
      const body = modal.querySelector(".kingsway-library-modal__body");
      const title = modal.querySelector(".kingsway-library-modal__title");
      const backBtn = modal.querySelector(".kingsway-library-modal__back");

      title.textContent = "Select Embed Type";
      backBtn.style.display = "none";

      body.innerHTML = `
        <div class="kingsway-library-select-type">
          <div class="kingsway-library-select-type__options">
            <div class="kingsway-library-select-type__option" data-type="video">
              <div class="kingsway-library-select-type__option-icon">
                <svg width="48" height="48" fill="none" viewBox="0 0 24 24">
                  <defs>
                    <clipPath id="video-clip">
                      <rect width="24" height="24" x="0" y="0" rx="0" />
                    </clipPath>
                  </defs>
                  <g clip-path="url(#video-clip)">
                    <g />
                    <g>
                      <rect
                        width="20"
                        height="14"
                        x="2"
                        y="5"
                        fill="none"
                        stroke="currentColor"
                        stroke-opacity="1"
                        stroke-width="1.5"
                        rx="2"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M6.25,19L6.25,5L7.75,5L7.75,19L6.25,19Z"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M16.25,19L16.25,5L17.75,5L17.75,19L16.25,19Z"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M6.5,10.25L2,10.25L2,8.75L6.5,8.75L6.5,10.25Z"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M6.5,15.25L2,15.25L2,13.75L6.5,13.75L6.5,15.25Z"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M21.5,10.25L17,10.25L17,8.75L21.5,8.75L21.5,10.25Z"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M21.5,15.25L17,15.25L17,13.75L21.5,13.75L21.5,15.25Z"
                      />
                    </g>
                    <g>
                      <path
                        fill="currentColor"
                        fill-opacity="1"
                        fill-rule="evenodd"
                        d="M14.4817,11.999220000000001Q14.52082,11.07812,13.78327,10.52496L12.55,9.6Q11.68475,8.9510643,10.717376,9.434752Q9.75,9.918441,9.75,11L9.75,12.69575Q9.75,13.72664,10.651655,14.226379999999999Q11.55331,14.72612,12.4275,14.17975L13.66077,13.40895Q14.44257,12.92033,14.4817,11.999220000000001ZM11.65,10.8L12.88327,11.72496Q12.98864,11.80398,12.98305,11.93557Q12.97746,12.06715,12.86577,12.13696L11.6325,12.90775Q11.50762,12.9858,11.378808,12.91441Q11.25,12.84302,11.25,12.69575L11.25,11Q11.25,10.84549,11.388197,10.77639Q11.52639,10.70729,11.65,10.8Z"
                      />
                    </g>
                  </g>
                </svg>
              </div>
              <div class="kingsway-library-select-type__option-title">Video Embed</div>
              <div class="kingsway-library-select-type__option-desc">Embed a single video player</div>
            </div>
            <div class="kingsway-library-select-type__option" data-type="widget">
              <div class="kingsway-library-select-type__option-icon">
                <svg width="48" height="48" viewBox="0 0 1024 1024">
                  <path
                    fill="currentColor"
                    d="M832 320H192a64.07 64.07 0 0 0-64 64v512a64.07 64.07 0 0 0 64 64h640a64.07 64.07 0 0 0 64-64V384a64.07 64.07 0 0 0-64-64z m0 576H192V384h640v512zM675.44 586.74l-192-127.91A64 64 0 0 0 384 512.09v255.82a64 64 0 0 0 99.49 53.26l192-127.91a64 64 0 0 0 0-106.52zM448 767.91V512.09L640 640zM256 96a32 32 0 0 1 32-32h448a32 32 0 0 1 0 64H288a32 32 0 0 1-32-32z m-64 128a32 32 0 0 1 32-32h576a32 32 0 0 1 0 64H224a32 32 0 0 1-32-32z"
                    p-id="2389"
                  />
                </svg>
              </div>
              <div class="kingsway-library-select-type__option-title">Widget Embed</div>
              <div class="kingsway-library-select-type__option-desc">Embed a video widget (swiper/float)</div>
            </div>
          </div>
        </div>
      `;

      body.querySelector('[data-type="video"]').onclick = onSelectVideo;
      body.querySelector('[data-type="widget"]').onclick = onSelectWidget;
    },

    renderVideoList(modal, data, options, callbacks) {
      const {
        page = 1,
        folderId = null,
        folderName = "",
        keyword = "",
      } = options;
      const { onSelect, onBack, onFolderClick, onPageChange, onSearch } =
        callbacks;

      const body = modal.querySelector(".kingsway-library-modal__body");
      const title = modal.querySelector(".kingsway-library-modal__title");
      const backBtn = modal.querySelector(".kingsway-library-modal__back");

      title.textContent = "Select Video";
      backBtn.style.display = "block";
      backBtn.onclick = onBack;

      // 重置存储，避免旧数据残留
      DataStore.reset("video");

      const totalPages = Math.ceil(
        (data.total || 0) / CONSTANTS.PAGE_SIZE.VIDEO
      );

      let html = this.buildVideoListHTML(data, options, totalPages);
      body.innerHTML = html;

      this.bindVideoListEvents(body, callbacks, options);
    },

    buildVideoListHTML(data, options, totalPages) {
      const { page, folderId, folderName, keyword } = options;

      let html = `
        <div class="kingsway-library-toolbar">
          <div class="kingsway-library-toolbar__search">
            <input type="text" placeholder="Search videos..." value="${keyword}" data-action="search-input" />
            <button class="kingsway-library-btn" data-action="search">Search</button>
          </div>
        </div>
      `;

      if (folderId) {
        html += `
          <div class="kingsway-library-breadcrumb">
            <span class="kingsway-library-breadcrumb__link" data-action="go-root">All Videos</span>
            <span class="kingsway-library-breadcrumb__sep">›</span>
            <span class="kingsway-library-breadcrumb__current">${folderName}</span>
          </div>
        `;
      }

      html += '<div class="kingsway-library-grid">';

      if (data.list?.length > 0) {
        data.list.forEach((item) => {
          html +=
            item.type === "folder"
              ? this.buildFolderCardHTML(item)
              : this.buildVideoCardHTML(item);
        });
      } else {
        html += '<div class="kingsway-library-empty">No videos found.</div>';
      }

      html += "</div>";

      if (totalPages > 1) {
        html += this.buildPaginationHTML(page, totalPages);
      }

      return html;
    },

    buildFolderCardHTML(item) {
      const coverItems = (item.dataList || [])
        .slice(0, 6)
        .map((v) => {
          const hasThumbnail = v.thumbnailUrl && v.thumbnailUrl.trim() !== "";
          const isFolder = v.type === "folder";
          let thumbnailUrl = hasThumbnail
            ? v.thumbnailUrl.replace(/^\s*`|`\s*$/g, "")
            : "";

          return `
          <div class="c-folder-cover__item ${
            isFolder ? "c-folder-cover__item--folder" : ""
          } ${isFolder && !hasThumbnail ? "c-folder-cover__item--empty" : ""}">
            <div class="c-folder-cover__item__inner">
              ${hasThumbnail ? `<img src="${thumbnailUrl}" alt="" />` : ""}
            </div>
          </div>
        `;
        })
        .join("");

      const folderId = DataStore.nextId("folder");
      DataStore.folders.set(folderId, item);

      return `
        <div class="c-folder-card" data-action="folder" data-folder-id="${folderId}">
          <svg viewBox="0 0 202 172" fill="none" xmlns="http://www.w3.org/2000/svg" preserveAspectRatio="none" class="c-folder-card__bg">
            <clipPath id="clip">
              <path d="M0,8 C0,3.58172 3.58172,0 8,0 L48.1211,0 C50.5509,0 54.1553838,0.560512007 56.1567838,1.93815001 L61.16268,4.73041737 C63.25898,6.17331737 67.8366041,6.14535472 70.3786041,6.26640472 L194,6.26640472 C198.418,6.26640472 202,9.74334883 202,14.1616488 L202,114 L202,164 C202,168.418 198.418,172 194,172 L8,172 C3.58172,172 0,168.418 0,164 L0,114 L0,10 L0,8 Z" id="clip-path"></path>
            </clipPath>
            <g id="folder-stroke">
              <path d="M0,8 C0,3.58172 3.58172,0 8,0 L48.1211,0 C50.5509,0 54.1553838,0.560512007 56.1567838,1.93815001 L61.16268,4.73041737 C63.25898,6.17331737 67.8366041,6.14535472 70.3786041,6.26640472 L194,6.26640472 C198.418,6.26640472 202,9.74334883 202,14.1616488 L202,114 L202,164 C202,168.418 198.418,172 194,172 L8,172 C3.58172,172 0,168.418 0,164 L0,114 L0,10 L0,8 Z"></path>
            </g>
            <filter id="blur">
              <feFlood flood-color="#fff" result="neutral"></feFlood>
              <feGaussianBlur in="SourceGraphic" stdDeviation="20" result="blurred"></feGaussianBlur>
              <feMerge>
                <feMergeNode in="neutral"></feMergeNode>
                <feMergeNode in="blurred"></feMergeNode>
              </feMerge>
            </filter>
            <g id="backdrop__L6jg2vGq3GKgzZp">
              <rect x="10" y="2" width="182" height="164" rx="8" ry="8" style="fill: rgb(179, 120, 250);"></rect>
            </g>
            <g style="clip-path: url('#clip');">
              <use xlink:href="#backdrop__L6jg2vGq3GKgzZp" style="filter: url('#blur');"></use>
            </g>
            <use xlink:href="#clip-path" style="fill: rgb(226, 230, 248);"></use>
          </svg>
          <div class="c-folder-card__cover">
            <div class="c-folder-cover">${coverItems}</div>
          </div>
          <div class="c-folder-card__row">
            <div class="c-folder-card__name">${item.folderName}</div>
          </div>
          <div class="c-folder-card__time">${Utils.formatDate(
            item.updateTime || 0
          )}</div>
        </div>
      `;
    },

    buildVideoCardHTML(item) {
      const thumbnailUrl = Utils.getThumbnailUrl(item);
      const duration = Utils.formatDuration(item.duration || 0);
      const date = Utils.formatDate(item.startTime || 0);

      const videoId = DataStore.nextId("video");
      DataStore.videos.set(videoId, item);

      return `
        <div class="c-video-card" data-action="video" data-video-id="${videoId}">
          <div class="c-video-card-cover-wrap">
            <img class="c-video-card__cover" src="${thumbnailUrl}" alt="${
              item.title
            }" />
            <div class="c-video-card-cover__mask"></div>
            <div class="c-video-card-detail__stats">
              <span class="c-video-card-detail__stats__item">${
                item.playCount || 0
              }</span>
            </div>
            <div class="c-video-card-cover__info">${duration}</div>
          </div>
          <div class="c-video-card-detail">
            <div class="c-vc-detail-row">
              <div class="c-vc-detail-title-box">
                <div class="c-vc-detail__title">${item.title}</div>
              </div>
            </div>
            <div class="c-video-card-detail__datetime">${date}</div>
          </div>
        </div>
      `;
    },

    buildPaginationHTML(page, totalPages) {
      return `
        <div class="kingsway-library-pagination">
          <button class="kingsway-library-btn" data-action="prev" ${
            page <= 1 ? "disabled" : ""
          }>Previous</button>
          <span class="kingsway-library-pagination__info">Page ${page} of ${totalPages}</span>
          <button class="kingsway-library-btn" data-action="next" ${
            page >= totalPages ? "disabled" : ""
          }>Next</button>
        </div>
      `;
    },

    bindVideoListEvents(body, callbacks, options) {
      const { onSelect, onFolderClick, onPageChange, onSearch } = callbacks;
      const { page } = options;

      body.querySelectorAll('[data-action="folder"]').forEach((el) => {
        el.onclick = () => {
          const folder = DataStore.folders.get(el.dataset.folderId);
          onFolderClick(folder);
        };
      });

      body.querySelectorAll('[data-action="video"]').forEach((el) => {
        el.onclick = () => {
          const video = DataStore.videos.get(el.dataset.videoId);
          onSelect(video);
        };
      });

      const searchInput = body.querySelector('[data-action="search-input"]');
      const searchBtn = body.querySelector('[data-action="search"]');
      if (searchBtn) {
        searchBtn.onclick = () => onSearch(searchInput.value);
      }
      if (searchInput) {
        searchInput.onkeypress = (e) => {
          if (e.key === "Enter") onSearch(searchInput.value);
        };
      }

      const goRoot = body.querySelector('[data-action="go-root"]');
      if (goRoot) goRoot.onclick = () => onFolderClick(null);

      const prevBtn = body.querySelector('[data-action="prev"]');
      const nextBtn = body.querySelector('[data-action="next"]');
      if (prevBtn) prevBtn.onclick = () => onPageChange(page - 1);
      if (nextBtn) nextBtn.onclick = () => onPageChange(page + 1);
    },

    renderWidgetList(modal, data, options, callbacks) {
      const { page = 1 } = options;
      const { onSelect, onBack, onPageChange } = callbacks;

      const body = modal.querySelector(".kingsway-library-modal__body");
      const title = modal.querySelector(".kingsway-library-modal__title");
      const backBtn = modal.querySelector(".kingsway-library-modal__back");

      title.textContent = "Select Widget";
      backBtn.style.display = "block";
      backBtn.onclick = onBack;

      // 重置存储，避免旧数据残留
      DataStore.reset("widget");

      const totalPages = Math.ceil(
        (data.total || 0) / CONSTANTS.PAGE_SIZE.WIDGET
      );

      let html = '<div class="kingsway-library-grid">';

      if (data.list?.length > 0) {
        data.list.forEach((widget) => {
          html += this.buildWidgetCardHTML(widget);
        });
      } else {
        html += '<div class="kingsway-library-empty">No widgets found.</div>';
      }

      html += "</div>";

      if (totalPages > 1) {
        html += this.buildPaginationHTML(page, totalPages);
      }

      body.innerHTML = html;

      body.querySelectorAll('[data-action="widget"]').forEach((el) => {
        el.onclick = () => {
          const widget = DataStore.widgets.get(el.dataset.widgetId);
          onSelect(widget);
        };
      });

      const prevBtn = body.querySelector('[data-action="prev"]');
      const nextBtn = body.querySelector('[data-action="next"]');
      if (prevBtn) prevBtn.onclick = () => onPageChange(page - 1);
      if (nextBtn) nextBtn.onclick = () => onPageChange(page + 1);
    },

    buildWidgetCardHTML(widget) {
      const thumbnailUrl =
        widget.thumbnailUrl ||
        "https://via.placeholder.com/300x200?text=No+Thumbnail";
      const title = widget.classify?.classifyName || "Untitled";
      const videoCount = widget.videoCount || 0;

      const widgetId = DataStore.nextId("widget");
      DataStore.widgets.set(widgetId, widget);

      return `
        <div class="c-video-card" data-action="widget" data-widget-id="${widgetId}">
          <div class="c-video-card-cover-wrap">
            <img class="c-video-card__cover" src="${thumbnailUrl}" alt="${title}" />
            <div class="c-video-card-cover__mask"></div>
            <div class="c-video-card-cover__info">${widget.type}</div>
          </div>
          <div class="c-video-card-detail">
            <div class="c-vc-detail-row">
              <div class="c-vc-detail-title-box">
                <div class="c-vc-detail__title">${title}</div>
              </div>
            </div>
            <div class="c-video-card-detail__datetime">Videos: ${videoCount}</div>
          </div>
        </div>
      `;
    },
  };

  // ============ 弹窗状态管理模块 ============
  const ModalState = {
    createState(videoOptions, widgetOptions) {
      return {
        currentType: null,
        videoOptions,
        widgetOptions,
        video: {
          page: 1,
          folderId: null,
          folderName: "",
          keyword: "",
        },
        widget: {
          page: 1,
        },
      };
    },

    reset(state) {
      state.currentType = null;
      state.video = { page: 1, folderId: null, folderName: "", keyword: "" };
      state.widget = { page: 1 };
    },

    resetVideo(state) {
      state.video = { page: 1, folderId: null, folderName: "", keyword: "" };
    },
  };

  // ============ 弹窗控制器模块 ============
  const ModalController = {
    async open(options = {}) {
      const {
        contentType = null,
        videoOptions = {},
        widgetOptions = {},
        onSelect,
        onClose,
      } = options;

      const modal = UIRenderer.createModal();
      const state = ModalState.createState(videoOptions, widgetOptions);
      state.currentType = contentType;

      const closeModal = () => {
        modal.remove();
        if (onClose) onClose();
      };

      modal.querySelector(".kingsway-library-modal__close").onclick =
        closeModal;
      modal.querySelector(".kingsway-library-modal__overlay").onclick =
        closeModal;

      // 检查API Key
      const key = await Api.getApiKey();
      if (!key) {
        UIRenderer.renderNoApiKey(modal, async () => {
          Config.clearApiKey();
          const newKey = await Api.getApiKey();
          if (newKey) {
            this.showContent(modal, state, onSelect, closeModal);
          }
        });
        return;
      }

      this.showContent(modal, state, onSelect, closeModal);
    },

    showContent(modal, state, onSelect, closeModal) {
      if (state.currentType === "video") {
        this.showVideoList(modal, state, onSelect, closeModal);
      } else if (state.currentType === "widget") {
        this.showWidgetList(modal, state, onSelect, closeModal);
      } else {
        this.showSelectType(modal, state, onSelect, closeModal);
      }
    },

    showSelectType(modal, state, onSelect, closeModal) {
      UIRenderer.renderSelectType(
        modal,
        () => {
          state.currentType = "video";
          this.showVideoList(modal, state, onSelect, closeModal);
        },
        () => {
          state.currentType = "widget";
          this.showWidgetList(modal, state, onSelect, closeModal);
        }
      );
    },

    async showVideoList(modal, state, onSelect, closeModal) {
      const body = modal.querySelector(".kingsway-library-modal__body");
      // If this is not the first render, keep existing height/content and show an overlay loader.
      // This prevents the modal from "shrinking" during pagination/loading.
      if (body && body.children && body.children.length > 0) {
        UIRenderer.setBodyLoading(body, true);
      } else {
        body.innerHTML =
          '<div class="kingsway-library-modal__loading">Loading...</div>';
      }

      const { video, videoOptions } = state;
      try {
        const data = await Api.fetchVideoList(
          video.page,
          video.folderId,
          video.keyword
        );

        UIRenderer.renderVideoList(
          modal,
          data,
          {
            page: video.page,
            folderId: video.folderId,
            folderName: video.folderName,
            keyword: video.keyword,
          },
          {
            onSelect: (videoItem) => {
              const html = HtmlGenerator.generateVideoHtml(
                videoItem,
                videoOptions
              );
              if (onSelect) onSelect("video", videoItem, html);
              closeModal();
            },
            onBack: () => {
              ModalState.reset(state);
              this.showSelectType(modal, state, onSelect, closeModal);
            },
            onFolderClick: async (folder) => {
              video.page = 1;
              if (folder) {
                video.folderId = folder.folderId || folder.id || null;
                video.folderName = folder.folderName || folder.name || "";
              } else {
                video.folderId = null;
                video.folderName = "";
              }
              video.keyword = "";
              await this.showVideoList(modal, state, onSelect, closeModal);
            },
            onPageChange: async (page) => {
              video.page = page;
              await this.showVideoList(modal, state, onSelect, closeModal);
            },
            onSearch: async (keyword) => {
              video.page = 1;
              video.keyword = keyword;
              await this.showVideoList(modal, state, onSelect, closeModal);
            },
          }
        );
      } finally {
        UIRenderer.setBodyLoading(body, false);
      }
    },

    async showWidgetList(modal, state, onSelect, closeModal) {
      const body = modal.querySelector(".kingsway-library-modal__body");
      if (body && body.children && body.children.length > 0) {
        UIRenderer.setBodyLoading(body, true);
      } else {
        body.innerHTML =
          '<div class="kingsway-library-modal__loading">Loading...</div>';
      }

      const { widget, widgetOptions } = state;
      try {
        const data = await Api.fetchWidgetList(widget.page);

        UIRenderer.renderWidgetList(
          modal,
          data,
          { page: widget.page },
          {
            onSelect: (widgetItem) => {
              const html = HtmlGenerator.generateWidgetHtml(
                widgetItem,
                widgetOptions
              );
              if (onSelect) onSelect("widget", widgetItem, html);
              closeModal();
            },
            onBack: () => {
              ModalState.reset(state);
              this.showSelectType(modal, state, onSelect, closeModal);
            },
            onPageChange: async (page) => {
              widget.page = page;
              await this.showWidgetList(modal, state, onSelect, closeModal);
            },
          }
        );
      } finally {
        UIRenderer.setBodyLoading(body, false);
      }
    },
  };

  // ============ 自动初始化 ============
  // 如果 PHP 已经通过 wp_localize_script 传递了配置，立即初始化
  // 这样可以确保在首次加载时使用正确的环境配置
  if (typeof window !== "undefined" && window.kingswayPlugin) {
    Config.init({
      apiRoot: window.kingswayPlugin.apiRoot || "",
      nonce: window.kingswayPlugin.nonce || "",
      env: window.kingswayPlugin.env || "production",
    });
  }

  // ============ 公共API ============
  return {
    init(options) {
      Config.init(options);
    },
    open(options) {
      return ModalController.open(options);
    },
    getApiKey() {
      return Api.getApiKey();
    },
    fetchVideoList(page, folderId, keyword) {
      return Api.fetchVideoList(page, folderId, keyword);
    },
    fetchWidgetList(page) {
      return Api.fetchWidgetList(page);
    },
    generateVideoHtml(video, options) {
      return HtmlGenerator.generateVideoHtml(video, options);
    },
    generateWidgetHtml(widget, options) {
      return HtmlGenerator.generateWidgetHtml(widget, options);
    },
    loadPlayerScript(type) {
      ScriptLoader.loadPlayerScript(type);
    },
    formatDuration(duration) {
      return Utils.formatDuration(duration);
    },
    changeEnv(newEnv) {
      Config.setEnv(newEnv);
    },
    getEnv() {
      return Config.getEnv();
    },
    getSdkUrl() {
      return Config.getSdkUrl();
    },
  };
});
